/*
 * Decompiled with CFR 0.152.
 */
package org.conscrypt.ct;

import java.security.cert.CertificateEncodingException;
import java.security.cert.CertificateException;
import java.security.cert.X509Certificate;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import org.conscrypt.NativeCrypto;
import org.conscrypt.OpenSSLX509Certificate;
import org.conscrypt.ct.CertificateEntry;
import org.conscrypt.ct.LogInfo;
import org.conscrypt.ct.LogStore;
import org.conscrypt.ct.Serialization;
import org.conscrypt.ct.SerializationException;
import org.conscrypt.ct.SignedCertificateTimestamp;
import org.conscrypt.ct.VerificationResult;
import org.conscrypt.ct.VerifiedSCT;

public class Verifier {
    private final LogStore store;

    public Verifier(LogStore store) {
        this.store = store;
    }

    public VerificationResult verifySignedCertificateTimestamps(List<X509Certificate> chain, byte[] tlsData, byte[] ocspData) throws CertificateEncodingException {
        OpenSSLX509Certificate[] certs = new OpenSSLX509Certificate[chain.size()];
        int i = 0;
        for (X509Certificate cert : chain) {
            certs[i++] = OpenSSLX509Certificate.fromCertificate(cert);
        }
        return this.verifySignedCertificateTimestamps(certs, tlsData, ocspData);
    }

    public VerificationResult verifySignedCertificateTimestamps(OpenSSLX509Certificate[] chain, byte[] tlsData, byte[] ocspData) throws CertificateEncodingException {
        if (chain.length == 0) {
            throw new IllegalArgumentException("Chain of certificates mustn't be empty.");
        }
        OpenSSLX509Certificate leaf = chain[0];
        VerificationResult result = new VerificationResult();
        List<SignedCertificateTimestamp> tlsScts = this.getSCTsFromTLSExtension(tlsData);
        this.verifyExternalSCTs(tlsScts, leaf, result);
        List<SignedCertificateTimestamp> ocspScts = this.getSCTsFromOCSPResponse(ocspData, chain);
        this.verifyExternalSCTs(ocspScts, leaf, result);
        List<SignedCertificateTimestamp> embeddedScts = this.getSCTsFromX509Extension(chain[0]);
        this.verifyEmbeddedSCTs(embeddedScts, chain, result);
        return result;
    }

    private void verifyEmbeddedSCTs(List<SignedCertificateTimestamp> scts, OpenSSLX509Certificate[] chain, VerificationResult result) {
        if (scts.isEmpty()) {
            return;
        }
        CertificateEntry precertEntry = null;
        if (chain.length >= 2) {
            OpenSSLX509Certificate leaf = chain[0];
            OpenSSLX509Certificate issuer = chain[1];
            try {
                precertEntry = CertificateEntry.createForPrecertificate(leaf, issuer);
            }
            catch (CertificateException certificateException) {
                // empty catch block
            }
        }
        if (precertEntry == null) {
            this.markSCTsAsInvalid(scts, result);
            return;
        }
        this.verifySCTs(scts, precertEntry, result);
    }

    private void verifyExternalSCTs(List<SignedCertificateTimestamp> scts, OpenSSLX509Certificate leaf, VerificationResult result) {
        CertificateEntry x509Entry;
        if (scts.isEmpty()) {
            return;
        }
        try {
            x509Entry = CertificateEntry.createForX509Certificate(leaf);
        }
        catch (CertificateException e) {
            this.markSCTsAsInvalid(scts, result);
            return;
        }
        this.verifySCTs(scts, x509Entry, result);
    }

    private void verifySCTs(List<SignedCertificateTimestamp> scts, CertificateEntry certEntry, VerificationResult result) {
        for (SignedCertificateTimestamp sct : scts) {
            VerifiedSCT.Builder builder = new VerifiedSCT.Builder(sct);
            LogInfo log = this.store.getKnownLog(sct.getLogID());
            if (log == null) {
                builder.setStatus(VerifiedSCT.Status.UNKNOWN_LOG);
            } else {
                VerifiedSCT.Status status = log.verifySingleSCT(sct, certEntry);
                builder.setStatus(status);
                if (status == VerifiedSCT.Status.VALID) {
                    builder.setLogInfo(log);
                }
            }
            result.add(builder.build());
        }
    }

    private void markSCTsAsInvalid(List<SignedCertificateTimestamp> scts, VerificationResult result) {
        for (SignedCertificateTimestamp sct : scts) {
            VerifiedSCT.Builder builder = new VerifiedSCT.Builder(sct);
            result.add(builder.setStatus(VerifiedSCT.Status.INVALID_SCT).build());
        }
    }

    private static List<SignedCertificateTimestamp> getSCTsFromSCTList(byte[] data, SignedCertificateTimestamp.Origin origin) {
        byte[][] sctList;
        if (data == null) {
            return Collections.emptyList();
        }
        try {
            sctList = Serialization.readList(data, 2, 2);
        }
        catch (SerializationException e) {
            return Collections.emptyList();
        }
        ArrayList<SignedCertificateTimestamp> scts = new ArrayList<SignedCertificateTimestamp>();
        for (byte[] encodedSCT : sctList) {
            try {
                SignedCertificateTimestamp sct = SignedCertificateTimestamp.decode(encodedSCT, origin);
                scts.add(sct);
            }
            catch (SerializationException serializationException) {
                // empty catch block
            }
        }
        return scts;
    }

    private List<SignedCertificateTimestamp> getSCTsFromTLSExtension(byte[] data) {
        return Verifier.getSCTsFromSCTList(data, SignedCertificateTimestamp.Origin.TLS_EXTENSION);
    }

    private List<SignedCertificateTimestamp> getSCTsFromOCSPResponse(byte[] data, OpenSSLX509Certificate[] chain) {
        if (data == null || chain.length < 2) {
            return Collections.emptyList();
        }
        byte[] extData = NativeCrypto.get_ocsp_single_extension(data, "1.3.6.1.4.1.11129.2.4.5", chain[0].getContext(), chain[0], chain[1].getContext(), chain[1]);
        if (extData == null) {
            return Collections.emptyList();
        }
        try {
            return Verifier.getSCTsFromSCTList(Serialization.readDEROctetString(Serialization.readDEROctetString(extData)), SignedCertificateTimestamp.Origin.OCSP_RESPONSE);
        }
        catch (SerializationException e) {
            return Collections.emptyList();
        }
    }

    private List<SignedCertificateTimestamp> getSCTsFromX509Extension(OpenSSLX509Certificate leaf) {
        byte[] extData = leaf.getExtensionValue("1.3.6.1.4.1.11129.2.4.2");
        if (extData == null) {
            return Collections.emptyList();
        }
        try {
            return Verifier.getSCTsFromSCTList(Serialization.readDEROctetString(Serialization.readDEROctetString(extData)), SignedCertificateTimestamp.Origin.EMBEDDED);
        }
        catch (SerializationException e) {
            return Collections.emptyList();
        }
    }
}

